/*******************************************************************************

    FILE:       testflash.h
 
                Copyright (C) 2001-2008 Cambridge Silicon Radio Ltd.
                Confidential. http://www.csr.com.
 
    PURPOSE:    Defines the TestFlash dll API, used for the flash programming
                of CSR BlueCore devices.

*******************************************************************************/

#ifndef __INCTestFlashh
#define __INCTestFlashh

#define TESTFLASH_API(T) T _stdcall


#ifndef COMMON__TYPES_H
typedef unsigned char  uint8;
typedef signed char   int8;
typedef unsigned short uint16;
typedef signed short   int16;
typedef unsigned long int uint32;
typedef signed long int int32;
#endif /* COMMON__TYPES_H */


/* Return codes */
#define TFL_OK 0
#define TFL_ERROR -1

#define TFL_SPI_LPT 0x1
#define TFL_SPI_USB 0x2

#define TFL_CHIP 0
#define TFL_FILE 1

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*************************************************************************************
	
	Function :	    int32 flInit(int32 port, int32 xtal, int32 delays, int32 transport)

	Parameters :    port - 
                        Defines the physical port to use. An LPT port number starting
                        from 1, or the id number of a USB SPI device. Set to -1 to 
                        use the default (port 1 if LPT, first USB SPI device if USB).
                        NOTE: Default may be overriden by the "SPIPORT" environment
                        variable.
                        <p>NOTE: When using SPI over LPT, the driver only supports
                        "traditional" parallel ports, i.e. LPT1 connected to the 
                        Super I/O chip on the PC motherboard. Some users have reported 
                        success with some PCI LPT cards, but this is not supported by 
                        CSR. The USB SPI device can be used where an LPT port is not 
                        available, or if more than one SPI port is required.

					xtal - 
                        Specifies module's crystal frequency. Acceptable values are 
						10, 16, 26, 32. Higner values make code run slower, lower 
						values may cause instability. (Casiras use 16Mhz).

					delays - 
                        Delay used in driving the SPI port. 1 = fastest SPI speed, 
                        but may give unreliablilty. 2 = reasonable value, higer 
                        values may be better if chip is in deep sleep.

                    transport -
                        Defines the SPI transport to be used, where:
                            TFL_SPI_LPT = 1
                            TFL_SPI_USB = 2
                        Set to -1 to use the default (from SPITRANS environment
                        variable if present), otherwise LPT.

	Returns :		non-zero on failure, 0 on success.

	Description :	This function is used to create the appropriate host side objects 
					to enable flash programming via the SPI or gang programming 
					interface.<br>
					No communication with BlueCore is carried out.

*************************************************************************************/
TESTFLASH_API(int32) flInit(int32 port, int32 xtal, int32 delays, int32 transport);

/*************************************************************************************
    
    Function :      int32 flInitTrans(char const * const trans, int32 xtal, 
                                        int32 delays)

    Parameters :    trans - 
                        String of space separated transport options that define the
                        transport to use. Commonly used options include:
                            SPITRANS (The physical transport to use, e.g. LPT, USB)
                            SPIPORT (The port number)
                        E.g. for LPT1, trans would be "SPITRANS=LPT SPIPORT=1".
                        These options override any environment variables of the same
                        names.
                        The transport string may be one of those returned by 
                        flGetAvailableSpiPorts, which returns transport strings for
                        all available SPI ports.
                        <p>NOTE: When using SPI over LPT, the driver only supports
                        "traditional" parallel ports, i.e. LPT1 connected to the 
                        Super I/O chip on the PC motherboard. Some users have reported 
                        success with some PCI LPT cards, but this is not supported by 
                        CSR. The USB SPI device can be used where an LPT port is not 
                        available, or if more than one SPI port is required.
                                
					xtal - 
                        Specifies module's crystal frequency. Acceptable values are 
						10, 16, 26, 32. Higner values make code run slower, lower 
						values may cause instability. (Casiras use 16Mhz).

					delays - 
                        Delay used in driving the SPI port. 1 = fastest SPI speed, 
                        but may give unreliablilty. 2 = reasonable value, higer 
                        values may be better if chip is in deep sleep.

    Returns :       non-zero on failure, 0 on success.

    Description :   This function is used to create the appropriate host side objects 
					to enable flash programming via the SPI or gang programming 
					interface.<br>
					No communication with BlueCore is carried out.
                    
*************************************************************************************/
TESTFLASH_API(int32) flInitTrans(char const * const trans, int32 xtal, int32 delays);

/*************************************************************************************
	
    Function :      int32 flReadProgramFiles(char const * const filename)

	Parameters :    filename - 
                        String containing the directory and filename root. This
						function will generate the XPV and XDV extensions.

	Returns :		non-zero on failure, 0 on success.

	Description :	This function is used to load the firmware files into memory.

*************************************************************************************/
TESTFLASH_API(int32) flReadProgramFiles(char const * const filename);

/*************************************************************************************
	
	Function :	    int32 flProgramBlock(void)

	Parameters :	None

	Returns :		-1 if there is a general error within TestFlash (normally due to
                    an incorrect sequence of API calls), otherwise one of the codes 
                    as follows:
                        <table>
						    <tr><td>FLASH_ERROR_NONE                        <td> 0
						    <tr><td>FLASH_ERROR_DOWNLOAD_FAILED             <td> 1
						    <tr><td>FLASH_ERROR_VERIFY_FAILED               <td> 2
						    <tr><td>FLASH_ERROR_TIMED_OUT                   <td> 3
						    <tr><td>FLASH_ERROR_CRC_FAILED                  <td> 4
						    <tr><td>FLASH_ERROR_READBACK_FAILED             <td> 5
						    <tr><td>FLASH_ERROR_COULD_NOT_DOWNLOAD_PROG     <td> 6
						    <tr><td>FLASH_ERROR_COULD_NOT_STOP_XAP          <td> 7
						    <tr><td>FLASH_ERROR_BOOT_PROG_HALTED            <td> 8
						    <tr><td>FLASH_ERROR_ERASE_FAILED                <td> 9
						    <tr><td>FLASH_ERROR_XAP_ERROR                   <td> 10
						    <tr><td>FLASH_ERROR_UNKNOWN_CHIP_TYPE           <td> 11
						    <tr><td>FLASH_ERROR_BROADCAST_MIXED_CHIP_TYPES  <td> 12
						    <tr><td>FLASH_ERROR_OPEN_FILE                   <td> 13
						    <tr><td>FLASH_ERROR_NO_IMAGE                    <td> 14
						    <tr><td>FLASH_ERROR_BUSY                        <td> 15
						    <tr><td>FLASH_ERROR_NO_FLASH                    <td> 16
                            <tr><td>FLASH_ERROR_OOM (out of memory)         <td> 17
                        </table>

	Description :	This function is used to download the firmware to the device under 
					test. The function will block until the process has completed 
					successfully or an error occurs.

*************************************************************************************/
TESTFLASH_API(int32) flProgramBlock(void);

/*************************************************************************************
	
    Function :      int32 flErase(void)

    Parameters :	None

	Returns :		-1 if there is a general error within TestFlash (normally due to
                    an incorrect sequence of API calls), otherwise one of the codes 
                    as follows:
                        <table>
						    <tr><td>FLASH_ERROR_NONE                        <td> 0
						    <tr><td>FLASH_ERROR_DOWNLOAD_FAILED             <td> 1
						    <tr><td>FLASH_ERROR_VERIFY_FAILED               <td> 2
						    <tr><td>FLASH_ERROR_TIMED_OUT                   <td> 3
						    <tr><td>FLASH_ERROR_CRC_FAILED                  <td> 4
						    <tr><td>FLASH_ERROR_READBACK_FAILED             <td> 5
						    <tr><td>FLASH_ERROR_COULD_NOT_DOWNLOAD_PROG     <td> 6
						    <tr><td>FLASH_ERROR_COULD_NOT_STOP_XAP          <td> 7
						    <tr><td>FLASH_ERROR_BOOT_PROG_HALTED            <td> 8
						    <tr><td>FLASH_ERROR_ERASE_FAILED                <td> 9
						    <tr><td>FLASH_ERROR_XAP_ERROR                   <td> 10
						    <tr><td>FLASH_ERROR_UNKNOWN_CHIP_TYPE           <td> 11
						    <tr><td>FLASH_ERROR_BROADCAST_MIXED_CHIP_TYPES  <td> 12
						    <tr><td>FLASH_ERROR_OPEN_FILE                   <td> 13
						    <tr><td>FLASH_ERROR_NO_IMAGE                    <td> 14
						    <tr><td>FLASH_ERROR_BUSY                        <td> 15
						    <tr><td>FLASH_ERROR_NO_FLASH                    <td> 16
                            <tr><td>FLASH_ERROR_OOM (out of memory)         <td> 17
                        </table>

    Description :   This function is used to erase the flash.

*************************************************************************************/
TESTFLASH_API(int32) flErase(void);

/*************************************************************************************
	
    Function :	    int32 flGangProgramBlock(uint16 device_mask, uint8 eraseFirst, 
                                                uint8 restartAfter, uint8 skipUnused)

	Parameters :	device_mask - 
                        Specifies which devices will be programmed. Uses a bit-field 
                        so 5 = 0000000000000101 = devices 0 and 2. A value of 0 (zero) 
                        will auto-detect and program all devices present.

					eraseFirst - 
                        Set to 1 to erase the entire flash of all devices before 
						programming. Set to 0 if erase is not required.

					restartAfter - 
                        Set to 1 to restart the XAP after completion of the flash 
                        process. Set to 0 if restart is not required.

					skipUnused - 
                        Set to 1 to skip attempts to stop and restart devices at 
						unused positions. Set to zero to attempt to stop and restart
                        all positions regardless of device_mask value (this takes
                        some time).

	Returns :		-1 if there is a general error within TestFlash (normally due to
                    an incorrect sequence of API calls), otherwise one of the codes 
                    as follows:
                        <table>
						    <tr><td>FLASH_ERROR_NONE                        <td> 0
						    <tr><td>FLASH_ERROR_DOWNLOAD_FAILED             <td> 1
						    <tr><td>FLASH_ERROR_VERIFY_FAILED               <td> 2
						    <tr><td>FLASH_ERROR_TIMED_OUT                   <td> 3
						    <tr><td>FLASH_ERROR_CRC_FAILED                  <td> 4
						    <tr><td>FLASH_ERROR_READBACK_FAILED             <td> 5
						    <tr><td>FLASH_ERROR_COULD_NOT_DOWNLOAD_PROG     <td> 6
						    <tr><td>FLASH_ERROR_COULD_NOT_STOP_XAP          <td> 7
						    <tr><td>FLASH_ERROR_BOOT_PROG_HALTED            <td> 8
						    <tr><td>FLASH_ERROR_ERASE_FAILED                <td> 9
						    <tr><td>FLASH_ERROR_XAP_ERROR                   <td> 10
						    <tr><td>FLASH_ERROR_UNKNOWN_CHIP_TYPE           <td> 11
						    <tr><td>FLASH_ERROR_BROADCAST_MIXED_CHIP_TYPES  <td> 12
						    <tr><td>FLASH_ERROR_OPEN_FILE                   <td> 13
						    <tr><td>FLASH_ERROR_NO_IMAGE                    <td> 14
						    <tr><td>FLASH_ERROR_BUSY                        <td> 15
						    <tr><td>FLASH_ERROR_NO_FLASH                    <td> 16
                            <tr><td>FLASH_ERROR_OOM (out of memory)         <td> 17
                        </table>
					
					If all the specified devices were programmed correctly the function 
					will return FLASH_ERROR_NONE, if there were 1 or more programming 
					failures then the function will return 
					FLASH_ERROR_COULD_NOT_DOWNLOAD_PROG. All other error returns denote 
					a gross failure.

	Description :	This function is used to download the firmware to up to 16 devices 
					when using the gang programmer reference device. This function 
					will block execution until the process completes successfully or 
					an error occurs.

					On completion with an error return of either FLASH_ERROR_NONE or 
					FLASH_ERROR_COULD_NOT_DOWNLOAD_PROG the programmer should call 
					get_bit_error_field() to determine which device has failed to 
					program correctly.

*************************************************************************************/
TESTFLASH_API(int32) flGangProgramBlock(uint16 device_mask, uint8 eraseFirst, 
                                        uint8 restartAfter, uint8 skipUnused);

/*************************************************************************************
	
	Function :	    uint16 flGetBitErrorField(void)

	Parameters :	None

	Returns :		Bitfield refering to the devices present on the gang programmer 
					and those that have failed. Will return 
                    (failed_devices & device_mask). 
					Therefore a return of zero (0) means all devices have been 
					programmed successfully. A return value of five (5) denotes that 
					devices 0 and 2 have failed and the remainder have successfully 
					been programmed.

	Description :	This function is used to determine which devices have failed to be 
					programmed during the gang programming process.

*************************************************************************************/
TESTFLASH_API(uint16) flGetBitErrorField(void);

/*************************************************************************************
	
	Function :	    int32 flProgramSpawn(void)

	Parameters :	None

	Returns :		-1 if there is a general error within TestFlash (normally due to
                    an incorrect sequence of API calls), otherwise one of the codes 
                    as follows:
                        <table>
						    <tr><td>FLASH_ERROR_NONE                        <td> 0
						    <tr><td>FLASH_ERROR_NO_IMAGE                    <td> 14
						    <tr><td>FLASH_ERROR_BUSY                        <td> 15
                        </table>

	Description :	This function is used to download the firmware to the device under 
					test. The function will spawn the process which performs the flash 
					programming. 
                    <p>NOTE: The return value indicates only whether the programming 
                    process started successfully or not. When the programming process
                    finishes, flGetLastError() should be called to get the final
                    status. flGetProgress() should be used to check for completion.

*************************************************************************************/
TESTFLASH_API(int32) flProgramSpawn(void);

/*************************************************************************************
	
	Function :	    int32 flGangProgramSpawn(uint16 device_mask, uint8 eraseFirst, 
                                                uint8 restartAfter, uint8 skipUnused)

	Parameters :	device_mask - 
                        Specifies which devices will be programmed. Uses a bit-field 
                        so 5 = 0000000000000101 = devices 0 and 2. A value of 0 (zero) 
                        will auto-detect and programme all devices present.

					eraseFirst - 
                        Set to 1 to erase the entire flash of all devices before 
						programming. Set to 0 if erase is not required.

					restartAfter - 
                        Set to 1 to restart the XAP after completion of the flash 
                        process. Set to 0 if restart is not required.

					skipUnused - 
                        Set to 1 to skip attempts to stop and restart devices at 
						unused positions. Set to zero to attempt to stop and restart
                        all positions regardless of device_mask value (this takes
                        some time).

	Returns :		-1 if there is a general error within TestFlash (normally due to
                    an incorrect sequence of API calls), otherwise one of the codes 
                    as follows:
                        <table>
						    <tr><td>FLASH_ERROR_NONE                        <td> 0
						    <tr><td>FLASH_ERROR_NO_IMAGE                    <td> 14
						    <tr><td>FLASH_ERROR_BUSY                        <td> 15
                        </table>

	Description :	This function is used to download the firmware to one or more 
                    devices connected to the gang programmer hardware.The function 
                    will spawn the process which performs the flash programming.
                    <p>NOTE: The return value indicates only whether the programming 
                    process started successfully or not. When the programming process
                    finishes, flGetLastError() and  should be called to get the final
                    status. flGetBitErrorField() should then be called to determine
                    which, if any, of the devices failed. flGetProgress() can be used 
                    to check for completion.

*************************************************************************************/
TESTFLASH_API(int32) flGangProgramSpawn(uint16 device_mask, uint8 eraseFirst, 
                                        uint8 restartAfter, uint8 skipUnused);

/*************************************************************************************
	
	Function :	    int32 flGetProgress(void)

	Parameters :	None

	Returns :		Progress so far in the range 0 to 100. Where 100 indicates the 
					spawned programming process has returned with or without an error.

	Description :	This function is used to check the progress of the process spawned 
					by flProgramSpawn() or flGangProgramSpawn().

*************************************************************************************/
TESTFLASH_API(int32) flGetProgress(void);

/*************************************************************************************
	
	Function :	    int32 flGetLastError(void)

	Parameters :	None

	Returns :		One of the codes as follows:
                        <table>
						    <tr><td>FLASH_ERROR_NONE                        <td> 0
						    <tr><td>FLASH_ERROR_DOWNLOAD_FAILED             <td> 1
						    <tr><td>FLASH_ERROR_VERIFY_FAILED               <td> 2
						    <tr><td>FLASH_ERROR_TIMED_OUT                   <td> 3
						    <tr><td>FLASH_ERROR_CRC_FAILED                  <td> 4
						    <tr><td>FLASH_ERROR_READBACK_FAILED             <td> 5
						    <tr><td>FLASH_ERROR_COULD_NOT_DOWNLOAD_PROG     <td> 6
						    <tr><td>FLASH_ERROR_COULD_NOT_STOP_XAP          <td> 7
						    <tr><td>FLASH_ERROR_BOOT_PROG_HALTED            <td> 8
						    <tr><td>FLASH_ERROR_ERASE_FAILED                <td> 9
						    <tr><td>FLASH_ERROR_XAP_ERROR                   <td> 10
						    <tr><td>FLASH_ERROR_UNKNOWN_CHIP_TYPE           <td> 11
						    <tr><td>FLASH_ERROR_BROADCAST_MIXED_CHIP_TYPES  <td> 12
						    <tr><td>FLASH_ERROR_OPEN_FILE                   <td> 13
						    <tr><td>FLASH_ERROR_NO_IMAGE                    <td> 14
						    <tr><td>FLASH_ERROR_BUSY                        <td> 15
						    <tr><td>FLASH_ERROR_NO_FLASH                    <td> 16
                            <tr><td>FLASH_ERROR_OOM (out of memory)         <td> 17
                        </table>

	Description :	This function will return the last error generated by the flash 
					programming stack. If called after flGetProgress() returns 100 
                    it can be used to check for programming errors.

*************************************************************************************/
TESTFLASH_API(int32) flGetLastError(void);

/*************************************************************************************
	
    Function :      int32 flResetAndStart(void)

    Parameters :    None

    Returns :		non-zero on failure, 0 on success.

    Description :	This function is used to reset the device after the spawned 
                    programming funtion has completed.

*************************************************************************************/
TESTFLASH_API(int32) flResetAndStart(void);

/*************************************************************************************
	
	Function :	    void flClose(void)

	Parameters :	None

	Returns :		None

	Description :	Deletes all the memory associated with the flash programming 
					process.

*************************************************************************************/
TESTFLASH_API(void) flClose(void);

/*************************************************************************************
	
	Function :	    int32 flGetVersion(char * versionStr)

	Parameters :	versionStr - 
                        Pre-allocated string for receiving the version number.

	Returns :		Always returns 0.

	Description :	Gets the version string for this DLL.

*************************************************************************************/
TESTFLASH_API(int32) flGetVersion(char * versionStr);

/*************************************************************************************
  
    Function :      int32 flStopProcessor(void)

    Parameters :    None.

    Returns :		non-zero on failure, 0 on success.

    Description :	This function will stop the processor on the current SPI
                    connection.<br>
                    A valid SPI connection has to be made using flInit() for this 
                    function to complete successfully.

*************************************************************************************/
TESTFLASH_API(int32) flStopProcessor(void);

/*************************************************************************************
  
    Function :      uint32 flGetDownloadTime(void)

    Parameters :    None.

    Returns :		The time taken in milliseconds for the last download. Returns 0 
                    if an error occurred or if download has not completed.

    Description :	Fetches the last download time.

*************************************************************************************/
TESTFLASH_API(uint32) flGetDownloadTime(void);

/*************************************************************************************
 
    Function :      int32 flGetAvailableSpiPorts(uint16* maxLen, char* ports, 
                                                    char* trans, uint16* count)

    Parameters :    maxLen - 
                        Size of the arrays pointed to by the ports and trans 
                        parameters. If this parameter indicates that the ports or 
                        trans arrays are too small to store the complete strings, 
                        then the value is set to the size required (and error is 
                        returned).
                        If any other error occurs, this value is set to zero.

                    ports -
                        Pointer to an array of ASCII chars where the comma separated
                        list of available SPI port names will be stored. These are
                        readable names which could be used for user selection.

                    trans -
                        Pointer to an array of ASCII chars where the comma separated
                        list of SPI transport options for each of the available ports
                        will be stored. The transport options for a port can be 
                        passed directly into the initTestEngineSpiTrans function to 
                        open the port.

                    count -
                        This value is set to the number of available ports found.

    Returns :       non-zero on failure, 0 on success.

    Description :   This function is used to get a list of available SPI ports. A 
                    char array, pointed to by the ports parameter, is filled with 
                    a comma separated list of port names. A further char array, 
                    pointed to by the trans parameter, is filled with a comma 
                    separated list of the relevant transport option strings that
                    specify each available SPI port.
                    <p>If the maxLen parameter indicates that either char array is not
                    large enough to contain the strings, Error is returned and the
                    maxLen parameter is set to the size required for the arrays.
                    <p>If any other error occurs, the maxLen parameter is set to 0.
                    <p>This function can be used by an application to get a list
                    of available SPI ports with which to populate a drop down list
                    or other means of selection. The strings returned in the ports
                    parameter are in human readable format for display / selection.
                    The strings returned in the trans parameter can be passed directly
                    to the flInitTrans function to open the port.
                    <p>NOTE: When using SPI over LPT, the driver only supports
                    "traditional" parallel ports, i.e. LPT1 connected to the 
                    Super I/O chip on the PC motherboard. Some users have reported 
                    success with some PCI LPT cards, but this is not supported by 
                    CSR. This function will return all available LPT ports regardless 
                    of whether or not they are supported by the SPI over LPT driver. 
                    The USB SPI device can be used where an LPT port is not available, 
                    or if more than one SPI port is required.

    Example :

        uint16 maxLen(256);
        uint16 count(0);
        char* portsStr = new char[maxLen];
        char* transStr = new char[maxLen];
        vector<string> ports; // The human readable port strings (e.g. "LPT1")
        vector<string> trans; // The transport option strings (e.g. "SPITRANS=LPT SPIPORT=1")

        int32 status = flGetAvailableSpiPorts(&maxLen, portsStr, transStr, &count);
        if( status != TFL_OK && maxLen != 0 )
        {
            // Not enough space - resize the storage
            portsStr = new char[maxLen];
            transStr = new char[maxLen];
            status = flGetAvailableSpiPorts(&maxLen, portsStr, transStr, &count);
        }
        if( status != TFL_OK || count == 0 )
        {
            cout << "Error getting SPI ports, or none found" << endl;
            delete[] portsStr;
            delete[] transStr;
            return;
        }

        // Split up the comma separated strings of ports / transport options
        split(ports, portsStr, ','); // Use these for user selection (e.g. drop down list)
        split(trans, transStr, ','); // Use these to open a transport

        // Open the SPI port using the trans string
        // For the purposes of this example, we're just using the first in the list
        status = flInitTrans(trans.at(0).c_str(), 0, 0);

        if(status == TFL_OK)
        {
            cout << "Connected!" << endl;

            // Flash download code here

            flClose();
        }

        delete[] portsStr;
        delete[] transStr;

        return;

*************************************************************************************/
TESTFLASH_API(int32) flGetAvailableSpiPorts(uint16* maxLen, char* ports, char* trans, 
                                                uint16* count);

/*************************************************************************************
	
	Function :	    int32 flGetFirmwareIds(uint8 source, uint32 *loaderId, 
                                            char *loaderName, uint32 *stackId,
                                            char *stackName, uint16 maxNameLen)

	Parameters :	source -
                        Value indicating where the firmware id information should be
                        read from, where:
                        <table>
                            <tr><td>TFL_CHIP = 0 = Read from attached chip
                            <tr><td>TFL_FILE = 1 = Read from file
                        </table>

                    loaderId -
                        Location for receiving the loader id value.
    
                    loaderName - 
                        Pre-allocated string for receiving the loader name.

                    stackId -
                        Location for receiving the stack id value.

                    stackName - 
                        Pre-allocated string for receiving the stack name.

                    maxNameLen -
                        The length of the buffers used for the loaderName and 
                        stackName strings.

	Returns :		Non-zero on failure, 0 on success.

	Description :	Gets the firmware id information from an attached device or from
                    a pre-loaded file (loaded using flReadProgramFiles).

*************************************************************************************/
TESTFLASH_API(int32) flGetFirmwareIds(uint8 source, uint32 *loaderId,
                                        char *loaderName, uint32 *stackId,  
                                        char *stackName, uint16 maxNameLen);

/*************************************************************************************
	
	Function :	    int32 flVerify(void)

	Parameters :	None

	Returns :		-1 if there is a general error within TestFlash (normally due to
                    an incorrect sequence of API calls), otherwise one of the codes 
                    as follows:
                        <table>
						    <tr><td>FLASH_ERROR_NONE                        <td> 0
						    <tr><td>FLASH_ERROR_VERIFY_FAILED               <td> 2
                        </table>

                    <p>It is possible for other error codes (values > 0) to be 
                    returned, e.g. if there are transport problems. These error codes
                    will be from the set listed for flGetLastError().

	Description :	This function is used to verify the firmware in flash against the
                    file loaded using flReadProgramFiles(). Therefore, 
                    flReadProgramFiles() must be called prior to calling this 
                    function.

*************************************************************************************/
TESTFLASH_API(int32) flVerify(void);

/*************************************************************************************
	
	Function :	    int32 flGetFlashInfo(uint16 *sectors, uint16* sizeMbits, 
                                         uint32 *manId, uint32 *devId)

	Parameters :	sectors -
                        Location for receiving the number of flash sectors.

	                sizeMbits -
                        Location for receiving the size of the flash in MBits.

	                manId -
                        Location for receiving the manufacturer ID of the flash.

	                devId -
                        Location for receiving the device ID of the flash.

    Returns :		Non-zero on failure, 0 on success.

	Description :	Gets the flash information, including the number of sectors, size, 
                    manufacturer ID, and device ID.
                    <p>This function will fail if called when a threaded (spawned) 
                    operation is in progress.

*************************************************************************************/
TESTFLASH_API(int32) flGetFlashInfo(uint16 *sectors, uint16* sizeMbits, 
                                    uint32 *manId, uint32 *devId);


/* DEPRECATED_START */
#define init_flash               flInit
#define read_program_files       flReadProgramFiles
#define program_flash_block      flProgramBlock
#define erase_flash              flErase
#define gang_program_flash_block flGangProgramBlock
#define get_bit_error_field      flGetBitErrorField
#define program_flash_spawn      flProgramSpawn
#define gang_program_flash_spawn flGangProgramSpawn
#define get_spawned_progress     flGetProgress
#define get_spawned_error        flGetLastError
#define reset_and_start          flResetAndStart
#define close_flash              flClose
#define get_version              flGetVersion
/* DEPRECATED_END */

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* __INCTestFlashh */
