/*******************************************************************************

    FILE:       e2api.h
 
                Copyright (C) 2001-2008 Cambridge Silicon Radio Ltd.
                Confidential. http://www.csr.com.
 
    PURPOSE:    Defines the E2Api dll API, used for production EEPROM 
                programming via CSR BlueCore devices.

*******************************************************************************/

#ifndef E2API_H
#define E2API_H

#ifdef E2API_EXPORTS
#define E2API_API(T) _declspec(dllexport) T _stdcall
#else
#define E2API_API(T) _declspec(dllimport) T _stdcall
#endif /* E2DLL_EXPORTS */

#ifndef COMMON__TYPES_H
typedef unsigned char  uint8;
typedef signed char   int8;
typedef unsigned short uint16;
typedef signed short   int16;
typedef unsigned long int uint32;
typedef signed long int int32;
#endif /* COMMON__TYPES_H */


const int32 E2_PASS = 1;
const int32 E2_FAIL = 0;

#define E2_SPI_LPT 0x1
#define E2_SPI_USB 0x2

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*************************************************************************************
    
    Function :      int32 e2GetVersion(char * versionStr)

    Parameters :    versionStr - 
                        A pointer to a preallocated string that will have the
                        version string copied to it.

    Returns :       Always 1.

    Description :   Retrieves the current version string.

    Example :
        {
            char versionStr[255];

            e2GetVersion(versionStr);

            cout << "E2Api version = " << versionStr << endl;
        }

*************************************************************************************/
E2API_API(int32) e2GetVersion(char* versionStr);

/*******************************************************************************
    
    Function :      int32 e2Init(int32 transport, int32 port, uint16 device, 
                                    uint16 size, uint16 mask)

    Parameters :
                    transport -
                        Defines the SPI transport to be used, where:
                            E2_SPI_LPT = 1
                            E2_SPI_USB = 2
                        Set to -1 to use the default (from SPITRANS environment
                        variable if present), otherwise LPT.

                    port -
                        Defines the physical port to use. An LPT port number starting
                        from 1, or the id number of a USB SPI device. Set to -1 to 
                        use the default (port 1 if LPT, first USB SPI device if USB).
                        NOTE: Default may be overriden by the "SPIPORT" environment
                        variable.
                        <p>NOTE: When using SPI over LPT, the driver only supports
                        "traditional" parallel ports, i.e. LPT1 connected to the 
                        Super I/O chip on the PC motherboard. Some users have reported 
                        success with some PCI LPT cards, but this is not supported by 
                        CSR. The USB SPI device can be used where an LPT port is not 
                        available, or if more than one SPI port is required.

                    device - 
                        Number of the device on a multispi, or 0 if no multispi.

                    size - 
                        Defines the size of the EEPROM device in log2 bytes. 
                        E.g. for a 256 Kbit (32 KByte) device size would be 15 
                        (i.e. 2^15)

                    mask -
                        Address mask of the EEPROM device to program.

    Returns :       Returns 1 on successful initialisation, otherwise 0.

    Description :   This function is used to initialise the SPI interface and
                    locate an EEPROM device.
                    It must be called before any other operations can be
                    performed.

//*****************************************************************************/
E2API_API(int32) e2Init(int32 transport, int32 port, uint16 device, 
                        uint16 size, uint16 mask);

/*******************************************************************************
    
    Function :      int32 e2InitTrans(const char *trans, uint16 device, 
                                        uint16 size, uint16 mask)

    Parameters :
                    trans -
                        String of space separated transport options that define the
                        transport to use. Commonly used options include:
                            SPITRANS (The physical transport to use, e.g. LPT, USB)
                            SPIPORT (The port number)
                        E.g. for LPT1, trans would be "SPITRANS=LPT SPIPORT=1".
                        These options override any environment variables of the same
                        names.
                        The transport string may be one of those returned by 
                        e2GetAvailableSpiPorts, which returns transport strings for
                        all available SPI ports.
                        <p>NOTE: When using SPI over LPT, the driver only supports
                        "traditional" parallel ports, i.e. LPT1 connected to the 
                        Super I/O chip on the PC motherboard. Some users have reported 
                        success with some PCI LPT cards, but this is not supported by 
                        CSR. The USB SPI device can be used where an LPT port is not 
                        available, or if more than one SPI port is required.

                    device - 
                        Number of the device on a multispi, or 0 if no multispi.

                    size - 
                        Defines the size of the EEPROM device in log2 bytes. 
                        E.g. for a 256 Kbit (32 KByte) device size would be 15 
                        (i.e. 2^15)

                    mask -
                        Address mask of the EEPROM device to program.

    Returns :       Returns 1 on successful initialisation, otherwise 0.

    Description :   This function (or e2Init) is used to initialise the SPI 
                    interface and locate an EEPROM device.
                    e2InitTrans or e2Init must be called before any other 
                    operations can be performed.
                    <p>The transport string may be one of those returned by 
                    e2GetAvailableSpiPorts, which returns transport strings for
                    all available SPI ports.

//*****************************************************************************/
E2API_API(int32) e2InitTrans(const char *trans, uint16 device, uint16 size, 
                                uint16 mask);

/*******************************************************************************
    
    Function :      int32 e2Close(void)

    Parameters :    None
                
    Returns :       Returns 1 on successful shutdown, otherwise 0.

    Description :   This function closes the SPI interface.
                    It must be called before e2Init can be called again for a 
                    different device.

//*****************************************************************************/
E2API_API(int32) e2Close(void);

/*******************************************************************************
    
    Function :      uint32 e2GetDeviceAddress(void)

    Parameters :    None

    Returns :       Returns the device address.

    Description :   This function is used to find the address that the EEPROM
                    device was found at from the last call to e2Init.

//*****************************************************************************/
E2API_API(uint32) e2GetDeviceAddress(void);

/*******************************************************************************
    
    Function :      int32 e2GetProgress(void)

    Parameters :    None

    Returns :       Returns an integer value in the range 0 - 100 representing
                    the percentage of completion of the current operation.

    Description :   This function is used to find out how far the current
                    operation has progressed.
                    <p>Do not use this function to check if an operation has
                    completed as an operation may not have truly finished even
                    though it is at 100% progress. Use e2IsBusy to check if an
                    operation has completely finished.

//*****************************************************************************/
E2API_API(int32) e2GetProgress(void);

/*******************************************************************************
    
    Function :      int32 e2GetResult(void)

    Parameters :    None

    Returns :       If the previously completed operation was successful the
                    return value is 1. If the operation failed the return value 
                    is 0.

    Description :   This function is used to find the outcome of the previously
                    completed operation. To check if an operation has completed 
                    use the e2IsBusy function.

//*****************************************************************************/
E2API_API(int32) e2GetResult(void);

/*******************************************************************************
    
    Function :      int32 e2IsBusy(void)

    Parameters :    None

    Returns :       If performing an operation the return value is non-zero.
                    If no operation is being performed the return value is zero.

    Description :   This function is used to find if an operation is ongoing. 
                    Typically this function would be called periodically after 
                    a call to e2Program, e2Clear or e2Dump with the wait
                    parameter set to 0.

//*****************************************************************************/
E2API_API(int32) e2IsBusy(void);

/*******************************************************************************
    
    Function :      int32 e2Stop(void)

    Parameters :    None

    Returns :       Always returns 1.

    Description :   This function stops the current operation. The state of the
                    EEPROM device is undefined if an operation is stopped
                    before completion.

//*****************************************************************************/
E2API_API(int32) e2Stop(void);

/*******************************************************************************
    
    Function :      int32 e2Program(const char * filename, const uint8 wait, 
                                    const uint8 verify)

    Parameters :
                    filename -
                        The path to a file containing the image to write to the 
                        EEPROM device.
                
                    wait -
                        Defines whether the function will wait for the 
                        programming operation to finish before returning.
                        Set non-zero to wait, otherwise 0 to return immediately.
                        If this parameter is 0 you can use the e2IsBusy and 
                        e2GetProgress functions to check the progress of the
                        operation.

                    verify -
                        Defines whether the contents of the EEPROM after 
                        writing should be verified against the given file.
                        Set to true to verify, otherwise set to false.
                        Note that verification will cause the operation to take 
                        more time.

    Returns :       Returns 1 if the device is successfully programmed,
                    otherwise 0. If wait is zero the function will return 1.
                    The result of the actual operation has to be found with a 
                    call to e2GetResult when the operation has completed.

    Description :   This function is used to program the EEPROM device found in 
                    the last call to e2Init.
                    The function can optionally wait for the programming
                    operation to finish before returning.
                    <p>Note that if wait is set to 0 (so that the function 
                    returns straight away) any calls to a function that access 
                    a device will return 0 until the current operation has 
                    completed or a call to e2Stop is made.
                    <p>It is advisable to poll for completion of an operation 
                    not less than once every second to prevent slow execution of
                    the operation.

//*****************************************************************************/
E2API_API(int32) e2Program(const char * filename, const uint8 wait, 
                            const uint8 verify);

/*******************************************************************************
    
    Function :      int32 e2Clear(const uint8 pattern, const uint8 wait, 
                                const uint8 verify)

    Parameters :
                    pattern -
                        Value to which all byte locations in the EEPROM device 
                        are set to.

                    wait -
                        Defines whether the function will wait for the 
                        programming operation to finish before returning.
                        Set non-zero to wait, otherwise 0 to return immediately.
                        If this parameter is 0 you can use the e2IsBusy and
                        e2GetProgress functions to check the progress of the
                        operation.

                    verify -
                        Defines whether the contents of the EEPROM after 
                        clearing should be verified. Set to true to verify, 
                        otherwise set to false. Default is true. Note that 
                        verification will cause the operation to take more time.

    Returns :       Returns 1 if successfully cleared, otherwise 0.

    Description :   This function is used to clear the EEPROM device found in 
                    the last call to e2Init. Each byte location is set to the 
                    given pattern value.

//*****************************************************************************/
E2API_API(int32) e2Clear(const uint8 pattern, const uint8 wait, 
                        const uint8 verify);

/*******************************************************************************
    
    Function :      int32 e2Dump(const char * filename, const uint8 wait, 
                                const uint8 verify)

    Parameters :
                    filename -
                        Path to a file to dump the image contained in the 
                        EEPROM device.
                
                    wait -
                        Defines whether the function will wait for the 
                        programming operation to finish before returning.
                        Set non-zero to wait, otherwise 0 to return immediately.
                        If this parameter is 0 you can use the e2IsBusy and
                        e2GetProgress functions to check the progress of the
                        operation.

                    verify -
                        Defines whether the contents of the EEPROM after dumping 
                        should be verified against the dumped file.
                        Set to 1 to verify, otherwise set to false.
                        Default is true. Note that verification will cause the 
                        operation to take more time.

    Returns :       Returns 1 if successfully dumped, otherwise 0.

    Description :   This function is used to dump the contents of the EEPROM
                    device found in the last call to e2Init to a file.

//*****************************************************************************/
E2API_API(int32) e2Dump(const char * filename, const uint8 wait, 
                        const uint8 verify);

/*******************************************************************************
    
    Function :      int32 e2Verify(const char * filename, const uint8 wait)

    Parameters :
                    filename -
                        Path to a file containing an image to compare with the 
                        contents of the EEPROM device.

                    wait -
                        Defines whether the function will wait for the 
                        programming operation to finish before returning.
                        Set non-zero to wait, otherwise 0 to return immediately.
                        If this parameter is 0 you can use the e2IsBusy and
                        e2GetProgress functions to check the progress of the
                        operation.

    Returns :       Returns 1 if the contents of the EEPROM device match the 
                    image contained in the file, otherwise 0.

    Description :   This function is used to compare the contents of the EEPROM
                    device found in the last call to e2Init with the image
                    contained in the given file.

//*****************************************************************************/
E2API_API(int32) e2Verify(const char * filename, const uint8 wait);

/*******************************************************************************
    
    Function :      int32 e2CheckSize(void)

    Parameters :    None

    Returns :       Returns 1 if the size of the EEPROM device is at least as 
                    large as the size given in the last call to e2Init, 
                    otherwise 0.

    Description :   This function is used to verify that the size of the EEPROM
                    device found in the last call to e2Init is at least as large
                    as the size that was specified. It does this by attempting
                    to write to the last location in the device using the size
                    given to e2Init.
                    <p>Note that the function will return 1 if the device is
                    larger than that specified in e2Init.
                    <p><b>WARNING:</b> This function does not take account of
                    devices that ignore out-of-range address bits. This may
                    cause the function to report a device size as valid even if
                    it is smaller than specified in e2Init.

//*****************************************************************************/
E2API_API(int32) e2CheckSize(void);

/*******************************************************************************
    
    Function :      int32 e2WriteHeader(void)

    Parameters :    None


    Returns :       1 if the header was written successfully, otherwise 0 on
                    error.

    Description :   This function is used to write the special header - as
                    required by BlueCore firmware, to the EEPROM device.
                    <p>This function cannot be called without the device first
                    being initialised with a call to e2Init.
                   
//*****************************************************************************/
E2API_API(int32) e2WriteHeader(void);

/*******************************************************************************
    
    Function :      int32 e2SetPioLines(int32 scl, int32 sda, int32 wp)

    Parameters :   
                    scl - 
                        PIO assignment for the I2C-bus SCL signal

                    sda - 
                        PIO assignment for the I2C-bus SDA signal

                    wp - 
                        PIO assignment for the write-protect signal
                  
    Returns :       1 if values set successfully, otherwise 0 on error.

    Description :   This function is used to set the PIO assignments for the 
                    I2C-bus signals to those other thasn the defaults.
                   
//*****************************************************************************/
E2API_API(int32) e2SetPioLines(int32 scl, int32 sda, int32 wp);

/*************************************************************************************
 
    Function :      int32 e2GetAvailableSpiPorts(uint16* maxLen, char* ports, 
                                                    char* trans, uint16* count)

    Parameters :    maxLen - 
                        Size of the arrays pointed to by the ports and trans 
                        parameters. If this parameter indicates that the ports or 
                        trans arrays are too small to store the complete strings, 
                        then the value is set to the size required (and error is 
                        returned).
                        If any other error occurs, this value is set to zero.

                    ports -
                        Pointer to an array of ASCII chars where the comma separated
                        list of available SPI port names will be stored. These are
                        readable names which could be used for user selection.

                    trans -
                        Pointer to an array of ASCII chars where the comma separated
                        list of SPI transport options for each of the available ports
                        will be stored. The transport options for a port can be 
                        passed directly into the e2InitTrans 
                        function to open the port.

                    count -
                        This value is set to the number of available ports found.

    Returns :       <table>
                        <tr><td>0 = Error
                        <tr><td>1 = Success
                    </table>

    Description :   This function is used to get a list of available SPI ports. A 
                    char array, pointed to by the ports parameter, is filled with 
                    a comma separated list of port names. A further char array, 
                    pointed to by the trans parameter, is filled with a comma 
                    separated list of the relevant transport option strings that
                    specify each available SPI port.
                    <p>If the maxLen parameter indicates that either char array is not
                    large enough to contain the strings, Error is returned and the
                    maxLen parameter is set to the size required for the arrays.
                    <p>If any other error occurs, the maxLen parameter is set to 0.
                    <p>This function can be used by an application to get a list
                    of available SPI ports with which to populate a drop down list
                    or other means of selection. The strings returned in the ports
                    parameter are in human readable format for display / selection.
                    The strings returned in the trans parameter can be passed directly
                    to the e2InitTrans function to open the port.
                    <p>NOTE: When using SPI over LPT, the driver only supports
                    "traditional" parallel ports, i.e. LPT1 connected to the 
                    Super I/O chip on the PC motherboard. Some users have reported 
                    success with some PCI LPT cards, but this is not supported by 
                    CSR. This function will return all available LPT ports regardless 
                    of whether or not they are supported by the SPI over LPT driver. 
                    The USB SPI device can be used where an LPT port is not available, 
                    or if more than one SPI port is required.

    Example :

        uint16 maxLen(256);
        uint16 count(0);
        char* portsStr = new char[maxLen];
        char* transStr = new char[maxLen];
        vector<string> ports; // The human readable port strings (e.g. "LPT1")
        vector<string> trans; // The transport option strings (e.g. "SPITRANS=LPT SPIPORT=1")

        int32 status = e2GetAvailableSpiPorts(&maxLen, portsStr, transStr, &count);
        if( status != E2_PASS && maxLen != 0 )
        {
            // Not enough space - resize the storage
            portsStr = new char[maxLen];
            transStr = new char[maxLen];
            status = e2GetAvailableSpiPorts(&maxLen, portsStr, transStr, &count);
        }
        if( status != E2_PASS || count == 0 )
        {
            cout << "Error getting SPI ports, or none found" << endl;
            delete[] portsStr;
            delete[] transStr;
            return;
        }

        // Split up the comma separated strings of ports / transport options
        split(ports, portsStr, ','); // Use these for user selection (e.g. drop down list)
        split(trans, transStr, ','); // Use these to open a transport

        // Open the SPI port using the trans string
        // For the purposes of this example, we're just using the first in the list
        status = e2InitTrans(trans.at(0).c_str(), 0, 10, 3);

        if(status != E2_FAIL)
        {
            cout << "Connected!" << endl;

            // Perform EEPROM operations here

            e2Close();
        }

        delete[] portsStr;
        delete[] transStr;

        return;

*************************************************************************************/
E2API_API(int32) e2GetAvailableSpiPorts(uint16* maxLen, char* ports, char* trans, 
                                        uint16* count);


#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* E2API_H */
